#!/usr/bin/env python3
"""
MySQL and phpMyAdmin Clean Installation Script
Completely removes existing installations and sets up fresh with custom credentials
"""

import subprocess
import sys
import getpass
import time

def run_command(command, shell=True, check=True):
    """Execute a shell command and return the result"""
    try:
        result = subprocess.run(
            command,
            shell=shell,
            check=check,
            capture_output=True,
            text=True
        )
        return result.returncode == 0
    except subprocess.CalledProcessError as e:
        print(f"Error executing command: {e}")
        print(f"Output: {e.output}")
        return False

def print_step(message):
    """Print a formatted step message"""
    print(f"\n{'='*60}")
    print(f"  {message}")
    print(f"{'='*60}\n")

def uninstall_mysql_phpmyadmin():
    """Completely remove MySQL and phpMyAdmin"""
    print_step("Step 1: Stopping MySQL service")
    run_command("sudo systemctl stop mysql", check=False)

    print_step("Step 2: Removing phpMyAdmin")
    run_command("sudo apt-get purge -y phpmyadmin", check=False)
    run_command("sudo rm -rf /usr/share/phpmyadmin", check=False)
    run_command("sudo rm -rf /etc/phpmyadmin", check=False)

    print_step("Step 3: Removing MySQL packages")
    run_command(
        "sudo apt-get purge -y mysql-server mysql-client mysql-common "
        "mysql-server-core-* mysql-client-core-* mysql-server-8.0",
        check=False
    )

    print_step("Step 4: Removing MySQL configuration and data")
    run_command("sudo rm -rf /etc/mysql", check=False)
    run_command("sudo rm -rf /var/lib/mysql", check=False)
    run_command("sudo rm -rf /var/log/mysql", check=False)

    print_step("Step 5: Cleaning up leftover packages")
    run_command("sudo apt-get autoremove -y")
    run_command("sudo apt-get autoclean -y")

    print("✓ Complete removal finished!")

def install_mysql(username, password):
    """Install MySQL with provided credentials"""
    print_step("Step 6: Installing MySQL Server")

    # Set environment variable for non-interactive installation
    env_cmd = "export DEBIAN_FRONTEND=noninteractive && "

    # Install MySQL without prompts
    if not run_command(
        env_cmd + "sudo -E apt-get install -y mysql-server mysql-client"
    ):
        print("Failed to install MySQL")
        return False

    time.sleep(3)

    print_step("Step 7: Starting MySQL service")
    run_command("sudo systemctl start mysql")
    run_command("sudo systemctl enable mysql")

    time.sleep(2)

    print_step("Step 8: Configuring MySQL user and permissions")

    # Create SQL commands to set up the user
    sql_commands = f"""
    -- Drop users if they exist
    DROP USER IF EXISTS 'root'@'localhost';
    DROP USER IF EXISTS '{username}'@'localhost';

    -- Create root user with password
    CREATE USER 'root'@'localhost' IDENTIFIED WITH caching_sha2_password BY '{password}';
    GRANT ALL PRIVILEGES ON *.* TO 'root'@'localhost' WITH GRANT OPTION;

    -- Create custom admin user
    CREATE USER '{username}'@'localhost' IDENTIFIED WITH caching_sha2_password BY '{password}';
    GRANT ALL PRIVILEGES ON *.* TO '{username}'@'localhost' WITH GRANT OPTION;

    -- Create phpMyAdmin user
    CREATE USER 'phpmyadmin'@'localhost' IDENTIFIED WITH caching_sha2_password BY '{password}';
    GRANT ALL PRIVILEGES ON *.* TO 'phpmyadmin'@'localhost' WITH GRANT OPTION;

    -- Disable password validation if enabled
    UNINSTALL COMPONENT 'file://component_validate_password';

    FLUSH PRIVILEGES;
    """

    # Execute SQL commands
    sql_file = "/tmp/mysql_setup.sql"
    with open(sql_file, 'w') as f:
        f.write(sql_commands)

    if run_command(f"sudo mysql < {sql_file}", check=False):
        print("✓ MySQL users configured successfully!")
    else:
        print("Note: Some SQL commands may have failed, but continuing...")

    run_command(f"rm {sql_file}")
    return True

def install_phpmyadmin(password):
    """Install phpMyAdmin with provided credentials"""
    print_step("Step 9: Configuring phpMyAdmin installation")

    # Pre-configure phpMyAdmin installation
    debconf_commands = [
        f"echo 'phpmyadmin phpmyadmin/dbconfig-install boolean true' | sudo debconf-set-selections",
        f"echo 'phpmyadmin phpmyadmin/app-password-confirm password {password}' | sudo debconf-set-selections",
        f"echo 'phpmyadmin phpmyadmin/mysql/admin-pass password {password}' | sudo debconf-set-selections",
        f"echo 'phpmyadmin phpmyadmin/mysql/app-pass password {password}' | sudo debconf-set-selections",
        f"echo 'phpmyadmin phpmyadmin/reconfigure-webserver multiselect apache2' | sudo debconf-set-selections"
    ]

    for cmd in debconf_commands:
        run_command(cmd)

    print_step("Step 10: Installing phpMyAdmin")

    if not run_command(
        "export DEBIAN_FRONTEND=noninteractive && "
        "sudo -E apt-get install -y phpmyadmin php-mbstring php-zip php-gd php-json php-curl"
    ):
        print("Failed to install phpMyAdmin")
        return False

    print_step("Step 11: Configuring Apache for phpMyAdmin")

    # Enable mbstring
    run_command("sudo phpenmod mbstring")

    # Make sure Apache config includes phpMyAdmin
    apache_conf = "Include /etc/phpmyadmin/apache.conf"
    run_command(
        f"echo '{apache_conf}' | sudo tee -a /etc/apache2/apache2.conf > /dev/null",
        check=False
    )

    # Restart Apache
    run_command("sudo systemctl restart apache2")

    print("✓ phpMyAdmin installed successfully!")
    return True

def main():
    """Main installation script"""
    print("""
╔════════════════════════════════════════════════════════════╗
║   MySQL & phpMyAdmin Clean Installation Script            ║
║   This will COMPLETELY REMOVE existing installations      ║
╚════════════════════════════════════════════════════════════╝
    """)

    # Get credentials
    print("Please provide your desired credentials:")
    username = input("Username [anthro]: ").strip() or "anthro"

    while True:
        password = getpass.getpass("Password: ")
        password_confirm = getpass.getpass("Confirm password: ")

        if password == password_confirm:
            if len(password) < 8:
                print("⚠ Password should be at least 8 characters for security")
                continue
            break
        else:
            print("❌ Passwords don't match. Try again.\n")

    # Confirmation
    print(f"\n⚠ WARNING: This will completely remove all existing MySQL databases!")
    confirm = input("Type 'YES' to continue: ")

    if confirm != "YES":
        print("Installation cancelled.")
        sys.exit(0)

    # Execute installation steps
    print("\n🚀 Starting installation process...\n")

    try:
        uninstall_mysql_phpmyadmin()

        if not install_mysql(username, password):
            print("❌ MySQL installation failed!")
            sys.exit(1)

        if not install_phpmyadmin(password):
            print("❌ phpMyAdmin installation failed!")
            sys.exit(1)

        print_step("✅ Installation Complete!")
        print(f"""
╔════════════════════════════════════════════════════════════╗
║                  Installation Successful!                  ║
╚════════════════════════════════════════════════════════════╝

Your MySQL and phpMyAdmin setup is ready!

Access phpMyAdmin at: http://localhost/phpmyadmin

Login credentials:
  Username: {username}
  Password: [your password]

Alternative users also created:
  - root (same password)
  - phpmyadmin (same password)

All users have full admin privileges.
        """)

    except KeyboardInterrupt:
        print("\n\n❌ Installation interrupted by user")
        sys.exit(1)
    except Exception as e:
        print(f"\n❌ Unexpected error: {e}")
        sys.exit(1)

if __name__ == "__main__":
    if subprocess.run("id -u", shell=True, capture_output=True).stdout.strip() == b"0":
        print("❌ Don't run this script as root. It will use sudo when needed.")
        sys.exit(1)

    main()
